package idemia.ipv;


import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import idemia.ipv.pojo.data.NotificationPayload;
import lombok.SneakyThrows;
import org.apache.commons.codec.binary.Base64;
import org.eclipse.jetty.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import spark.Request;
import spark.Response;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.io.IOException;
import java.io.InputStream;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Properties;

import static spark.Spark.*;

/**
 * notification sample server!
 *
 */
public class NotificationDemo {
    static ObjectMapper objectMapper = new ObjectMapper().registerModule(new JavaTimeModule());

    static Logger logger = LoggerFactory.getLogger(NotificationDemo.class);

    static Configuration configuration = loadConfiguration();

    public static void main( String[] args )    {
        logger.info("start notification server");
        port(configuration.getPort());
        setupRoutes();
    }



    private static void setupRoutes() {
        setupBusinessRoutes();
        init();
    }


    private static void setupBusinessRoutes() {
        post(configuration.getUrlPath(), NotificationDemo::handleNotification);
    }

    private static String handleNotification(Request request, Response response) throws IOException {
        logger.info("received : "  +  request.body() + "\n");
        String hmac = request.headers(configuration.getHeaderHmac());
        if(!verifyHash(hmac, request.body())){
            logger.error("hash is invalid");
            return error(response);
        }
        getEvidenceFromNotication(request);
        return ok(response);
    }


    private static void getEvidenceFromNotication(Request request) throws IOException {
        NotificationPayload notificationPayload = objectMapper.readValue(request.body(), NotificationPayload.class);
        // get evidence after it
    }


    public static boolean verifyHash(String signature, String payload) {
        try {
            String secret = configuration.getSecret();
            String computedSignature = generateHmac(payload, secret, "HmacSHA256");
            return signature.equals(computedSignature);
        } catch (Exception ex) {
            logger.error("hash is invalid", ex);
            return false;
        }
    }

    public static String generateHmac(String payload, String secret, String algo)
            throws NoSuchAlgorithmException, InvalidKeyException {
        Mac sha256_HMAC = Mac.getInstance(algo);
        SecretKeySpec secret_key = new SecretKeySpec(secret.getBytes(), algo);
        sha256_HMAC.init(secret_key);
        return Base64.encodeBase64String(sha256_HMAC.doFinal(payload.getBytes()));
    }


    private static String ok(Response response) {
        response.type("application/json");
        response.status(HttpStatus.OK_200);
        return "ok";
    }


    private static String error(Response response) {
        response.type("application/json");
        response.status(HttpStatus.UNAUTHORIZED_401);
        return "hmac is on error";
    }




    @SneakyThrows
    static Configuration loadConfiguration() {
        try (InputStream input = NotificationDemo.class.getClassLoader().getResourceAsStream("configure.properties")) {
            Properties prop = new Properties();
            prop.load(input);
            return new Configuration(prop);
        }
    }

}
